from dotenv import load_dotenv
load_dotenv("/var/local/env/rdi")

import logging
from datetime import datetime
import os, re, json, math, random, requests, client
from logging.handlers import RotatingFileHandler
from flask import Flask, render_template, url_for, request, flash, session
from flask_socketio import SocketIO

# Creates an instance of the flask class
app = Flask(__name__)

# Creates an instance of socketio
socketio = SocketIO(app)
socketio.ping_timeout = 60
# Get admin portal url and port from environment variable
apiBaseUrl = f"{os.environ.get('API_PORTAL_URL')}:{os.environ.get('API_PORTAL_PORT')}"
# Generate and store the secret key for sessions
app.secret_key = os.urandom(12).hex()

# Load ssl certificates location into variable
ssl_context = (f"{os.environ.get('SSL_CERT')}", f"{os.environ.get('SSL_KEY')}")


def makeApiRequest(username, password):
    "Make a post request to the api server"

    # Get bridgename from settings.cfg file
    f = open("/var/local/env/settings.cfg", "r")
    raw_version_data = f.read()
    f.close()
    bridgeName = raw_version_data.split("\n")[0].split("=")[1]

    # Prepare parameters for post request
    payload = {'username': username.encode('utf-8'), 'password': password.encode('utf-8'), 'bridgeName': bridgeName.encode('utf-8')}
    url=f"{apiBaseUrl}/api/clients/bridge/detail"
    app.logger.debug(f"url: {url}, payload: {payload}")

    # Send post request
    response = requests.post(url = url,data=payload, verify=False)
    app.logger.debug(response.content)

    return response


def getSlaveClientDetails(username):
    "Make a post request to the api server"

    # Get bridgename from settings.cfg file
    f = open("/var/local/env/settings.cfg", "r")
    raw_version_data = f.read()
    f.close()
    bridgeName = raw_version_data.split("\n")[0].split("=")[1]

    # Prepare parameters for post request
    payload = {'username': username.encode('utf-8'), 'bridgeName': bridgeName.encode('utf-8')}
    url=f"{apiBaseUrl}/api/clients/bridge/slaveDetail"
    app.logger.debug(f"url: {url}, payload: {payload}")

    # Send post request
    response = requests.post(url = url,data=payload, verify=False)
    app.logger.debug(response.content)

    return response


def setAllowedPorts():
    "Set allowed ports on redis at startup"

    # Default allowed ports
    minPort = 6000
    maxPort = 8000

    # Read settings file
    try:
        with open("/var/local/env/settings.cfg", "r") as f:
            raw_data = f.read()
        app.logger.debug("Successfully read settings.cfg file.")
    except Exception as e:
        app.logger.error(f"Error reading settings.cfg file: {e}")
        raise

    # Get allowed ports from read file
    data = raw_data.split('\n')
    if (len(data) > 1):
        minPort = re.sub('\D', '', data[1])
    if (len(data) > 2):
        maxPort = re.sub('\D', '', data[2])

    # Store allowed ports on redis
    for i in range(int(minPort), int(maxPort)):
        client.sadd('docker_allowed_ports', i)


def getFreeDockerPort():
    "Get available port from redis"

    # Get a set of unused ports
    availablePorts = client.sdiff('docker_allowed_ports', 'docker_used_ports')

    # Return one port from set
    if (len(availablePorts) > 0):
        port = availablePorts.pop()
        return int(port)
    else:
        app.logger.error("No available port")
        return 0


def generatePassword():
    "Generate a random password of 8 characters"

    # Set parameters for password
    retVal = ""
    length = 8
    charset = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

    # Add random characters from charset
    n = len(charset)
    for i in range(1 , length):
        retVal += charset[(math.floor(random.random() * n))]
    return retVal


def resetPassword(username, newPassword, confirmPassword):
    "Send post request to api server to reset password"

    try:
        # Prepare parameters for post request
        app.logger.info("Starting reset Password POST request.")
        api_portal_url = f"{apiBaseUrl}/api/clients/bridge/resetPassword"
        payload = {'username': username, 'password': newPassword, 'conformPassword': confirmPassword}
        app.logger.debug(f"url: {api_portal_url}, payload: {payload}")

        # Send post request
        response = requests.post(url=api_portal_url,data=payload, verify=False)
        app.logger.debug(f'response: {response.content}')

        return {'success': response.ok, 'status': 200}
    except Exception as e:
        app.logger.error(e)
        return {'success': False, 'status': 401}


def stopAllDockers():
    "Stop all docker containers"

    try:
        # Check if there are any running containers
        running_containers = os.popen("docker ps -q").read().strip()
        if not running_containers:
            return {'success': True, 'status': 200}

        # Build the stop command
        stop_command = f"docker stop {running_containers}"
        app.logger.debug(f"DOCKER COMMAND: {stop_command}")

        # Run the stop command
        result = os.system(stop_command)

        if result == 0:
            return {'success': True, 'status': 200}
        else:
            return {'success': False, 'status': 401}

    except Exception as e:
        app.logger.error(e)
        return {'success': False, 'status': 401}

def stopDocker():
    "Stop the Docker container and remove it from redis"

    try:
        # Prepare stop docker command
        app.logger.info("Preparing the stop docker command")
        b2bId = client.get('b2bId').decode('utf-8')
        agent_id = client.get('agent_id').decode('utf-8')
        stopCommand = f"docker stop {b2bId}"
        app.logger.debug(f"DOCKER COMMAND:  {b2bId}")

        # Run stop docker command
        os.popen(stopCommand).read()

        # Remove docker from running_containers in redis
        client.delete(agent_id)
        client.hdel("running_containers", agent_id)
        return {'success': True, 'status': 200}
    except Exception as e:
        app.logger.error(e)
        return {'success': False, 'status': 401}


def getVersion():
    "Get version data from .version file at startup"

    # Declare global variables
    global major_release
    global minor_release
    global patch_number
    global build_number

    try:
        # Read the version file
        f = open("models/general/.version", "r")
        raw_version_data = f.read()
        f.close()

        # Store the version information into global variables
        data = raw_version_data.split('\n')
        major_release = data[0].split('=')[1]
        minor_release = data[1].split('=')[1]
        patch_number = data[2].split('=')[1]
        build_number = data[3].split('=')[1]

    except Exception as e:
        app.logger.error('Version file does not exist')
        return {'success': False, 'status': 401}


def b2b_validation():
    "Validate the b2b client and run docker container"

    try:
        # Send login request to api server
        response_raw = makeApiRequest(session['username'], session['password'])
        response = response_raw.json()

        # Return when change_password is recieved in api response
        if (response_raw.ok and 'change_password' in response and response['change_password'] == True):
            return {'change_password': True, 'status': 200}

        # Return when validation is unsuccesful
        if (response_raw.status_code == 401):
            return {'success': False, 'status': 200}

        # Store response data in session
        session['agent_id'] = response['agent_id']
        client.set('agent_id',session['agent_id'])
        session['serial_number'] = response['serial_number']
        session['device_hardware_id'] = response['device_hardware_id']

        test = client.get('agent_id')
        app.logger.debug(f"Session data updated: agent_id={session['agent_id']}, serial_number={session['serial_number']}, device_hardware_id={session['device_hardware_id']}")        
        # Return when MFA is enabled in portal
        if (response_raw.ok and 'client_mfa' in response and response['client_mfa'] == True):
            app.logger.info("MFA enabled for client.")
            return {'client_mfa': True, 'agent_id': session['agent_id'], 'serial_number': session['serial_number'], 'device_hardware_id': session['device_hardware_id'], 'status': 200}

        return start_docker()

    except Exception as error:
        app.logger.error(error)
        return {'success': False, 'status': 500}


def otp():
    "Validate the client provided OTP"

    try:
        # Set the parameters form api request
        url = f"{apiBaseUrl}/api/clients/verify/otp?username={session['username']}&token={session['otp']}"
        app.logger.debug(f"url: {url}")

        # Send the api request
        response = requests.get(url=url, verify=False)
        app.logger.debug(f"response: {response.content}")
        data = json.loads(response.content)

        # Return otp validation status
        if (response.status_code == 200 and data['success'] == True):
            return {"success": True, 'status': 200}
        else:
            return {"success": False, 'status': 200}

    except Exception as e:
        app.logger.error(e)
        return {"success": False, 'status': 500}


def start_docker():
    "Start the docker container"

    try:
        # Prepare the docker command
        port = getFreeDockerPort()
        if port == 0:
            raise Exception("No available port")

        vnc_password = generatePassword()
        docker_command = f"/var/local/scripts/b2b_container_create.sh  '{session['agent_id']}' '{port}' '{session['serial_number']}' '{session['username']}' '{session['password']}' 'invisily' '{vnc_password}'"

        # Run the docker command
        app.logger.debug(f'DOCKER COMMAND: {docker_command}')
        stdout = os.popen(docker_command).read()

        # Store container info in redis
        if stdout:
            client.set(session['agent_id'], stdout)
            client.sadd('docker_used_ports', port)

            ip = request.host.split(':')[0]
            ts = str(datetime.utcnow())
            client.hset("running_containers", session['agent_id'], ts)

            # Return the url for accessing docker container
            url = f"http://invisily:{vnc_password}@{ip}:{port}"
            return {'success': True, 'url': url, 'agent_id': session['agent_id'], 'id': stdout.replace('\n', ''), 'status': 200}

        else:
            raise Exception("Unable to run the docker command")

    except Exception as error:
        app.logger.error(error)
        return {'success': False, 'status': 500}


def attempt_limit_reached():
    "Block client after extensive login attempts"

    # Check if this the the first login attempt in a while
    if client.exists(session['username']):
        # Increment the attempt count and reset expiration time
        attempts = int(client.get(session['username'])) + 1
        client.setex(session['username'], 60, attempts)

        # Return true if the attempt limit was reached
        if attempts > 10:
            return True
    else:
        # Add username to redis with one minute expiration
        client.setex(session['username'], 60, '1')

    return False

def logging_handler():

    try:
        log_folder = '/var/log/invisily/rdi-bridge-portal/'

        if not os.path.exists(log_folder):
            os.makedirs(log_folder)

        log_file = os.path.join(log_folder, 'app.log')
        error_log_file = os.path.join(log_folder, 'error.log')

        # Create a rotating flask log file handler
        flask_log_handler = RotatingFileHandler(log_file, maxBytes=1024 * 1024, backupCount=5)
        flask_log_handler.setLevel(logging.DEBUG)

        # Create a rotating flask error log file handler
        flask_error_log_handler = RotatingFileHandler(error_log_file, maxBytes=1024 * 1024, backupCount=5)
        flask_error_log_handler.setLevel(logging.ERROR)

        flask_default_log = logging.getLogger()
        flask_default_log.addHandler(flask_log_handler)
        flask_default_log.addHandler(flask_error_log_handler)
        return {'success': True, 'status': 200}

    except Exception as error:
        app.logger.error(error)
        return {'success': False, 'status': 500}

@app.route('/form_signin', methods=['POST', 'GET'])
def form_signin():
    "Validate the client credentials"

    if request.method == 'POST':
        # Store login data in session
        session['username'] = request.form['username']
        session['password'] = request.form['password']

        client.set('username',session['username'])

        # Check if login attempt limit is reached
        if attempt_limit_reached():
            flash("Login attempt limit reached. Please try again later", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

        else:
            # Validate the login credentials
            response = b2b_validation()
            if response['status'] == 200:

                # Redirect to change password page if change_password field is True
                if 'change_password' in response and response["change_password"] == True:
                    return render_template("resetPassword.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

                # Redirect to otp page if mfa is enabled
                elif 'client_mfa' in response and response["client_mfa"] == True:
                    session['otpData'] = json.dumps(response)
                    return render_template("otp.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

                # Store response data in session if login is successful
                elif response["success"] == True:
                    session['b2bId'] = response['id']
                    client.set("b2bId",session["b2bId"])
                    session['agent_id'] = response['agent_id']
                    session['b2bURL'] = response['url']

                    # Redirect to keepalive page
                    flash("Please wait, While RDI Portal is starting up..")
                    return render_template("keepalive.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, b2b_url = session['b2bURL'],b2bId=session['b2bId'])

                else:
                    # Redirect to home page and show the error message
                    flash("Login failed: Invalid login credentials (Please try again)", 'error')
                    return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

            else:
                # Redirect to home page and show the error message
                flash("Something went wrong. Please try again later", 'error')
                return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
    else:
        # Refresh homepage if it is not a post request
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/form_signin_stage_1', methods=['POST', 'GET'])
def form_signin_stage_1():
    "Validate the client credentials"

    if request.method == 'POST':
        # Store login data in session
        session['username'] = request.form['username']
        # Validate the login credentials
        response = getSlaveClientDetails(session['username'])
        
        if response.status_code == 200:
            response = response.json()
            # Redirect to change password page if change_password field is True
            if 'change_password' in response and response["change_password"] == True:
                return render_template("resetPassword.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

            else:
                return render_template("index.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

        elif response.status_code == 404:
            # Show error message for non-existent email address
            flash("User not found. Please try again with valid username.", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

        else:
            # Redirect to home page and show the error message
            flash("Something went wrong. Please try again later", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
    else:
        # Refresh homepage if it is not a post request
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/otpConfirm', methods=['POST', 'GET'])
def otpConfirm():
    "Process the client provided otp"

    if request.method == 'POST':
        # Read the input otp
        session['otp'] = request.form['otp']

        # Validate the otp
        response = otp()

        # Continue if otp is valid
        if (response['success']):
            # Start the docker container
            response = start_docker()

            # Store the docker info in session
            if response["success"] == True:
                session['b2bId'] = response['id']
                session['b2bURL'] = response['url']
                session['agent_id'] = response['agent_id']

                # Redirect to keepalive page
                flash("Please wait, While RDI Portal is starting up..")
                return render_template("keepalive.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, b2b_url = session['b2bURL'])
            else:
                # Redirect to home page and show the error message
                flash("Something went wrong. Please try again later", 'error')
                return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

        else:
            # Redirect to home page and show the error message
            flash("Login failed: Invalid OTP (Please try again)", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
    else:
        # Redirect to homepage if it is not a post request
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/resetBtn', methods=['POST', 'GET'])
def resetBtn():
    "Send reset password request to api server"

    if request.method == 'POST':
        # Set the parameters form api request
        url = f"{apiBaseUrl}/api/clients/bridge/resetPasswordRequest"
        payload = {'email': request.form['email'].encode('utf-8')}
        app.logger.debug(f"url: {url}, payload: {payload}")

        # Send the api request
        response = requests.post(url=url, data=payload, verify=False)
        app.logger.debug(f"response: {response.content}")

        # Check if password change request was successful
        if response.status_code == 200:
            data = json.loads(response.content)
            if data["success"] == True:
                # Redirect to home page and show the success message
                flash("An email with reset password link has been sent", 'success')
                return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
            else:
                # Redirect to home page and show the error message
                flash("Something went wrong. Please try again later", 'error')
                return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
        elif response.status_code == 404:
                # Show error message for non-existent email address
                flash("Email address does not exist. Please try again", 'error')
                return render_template('reset.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
        else:
            # Redirect to home page and show the error message
            flash("Something went wrong. Please try again later", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

    else:
        # Redirect to homepage if it is not a post request
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/resetBtnForm', methods=['POST', 'GET'])
def resetBtnForm():
    "Update the client password"

    if request.method == 'POST':
        # Get the new password from client
        session['username'] = request.form['username']
        session['password'] = request.form['password']
        session['newPassword'] = request.form['newPassword']
        session['confirmPassword'] = request.form['confirmPassword']

        if not session['username'] or not session['password'] or not session['newPassword'] or not session['confirmPassword']:
            flash("Missing credentials", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

        # Check if password is not empty
        if (len(session['password']) == 0):
            flash("Password is required", 'error')
            return render_template('resetPassword.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])
        
        if (len(session['newPassword']) == 0):
            flash("New Password is required", 'error')
            return render_template('resetPassword.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

        # Check if confirm password is empty
        if (len(session['confirmPassword']) == 0):
            flash("Confirm Password is required", 'error')
            return render_template('resetPassword.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

        # Check if both passwords are equal
        if (session['newPassword'] != session['confirmPassword']):
            flash("Password do not match", 'error')
            return render_template('resetPassword.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

        # Check if password length is valid
        if (len(session['newPassword']) < 8 or len(session['newPassword']) > 16):
            flash("Password length should be between 8 to 16 characters", 'error')
            return render_template('resetPassword.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number, username=session['username'])

        # Validate the login credentials
        # Send login request to api server
        response_raw = makeApiRequest(session['username'], session['password'])
        response = response_raw.json()

        # Return when change_password is recieved in api response
        if (response_raw.ok and 'change_password' in response and response['change_password'] == True):

            # Send post request to api server to reset password
            response = resetPassword(session['username'], session['newPassword'], session['confirmPassword'])

            if response['status'] == 200:
                if (response["success"] == True):
                    # Redirect to home page and show the success message
                    flash('Password has been updated successfully', 'success')
                    return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
                else:
                    # Redirect to home page and show the error message
                    flash("Something went wrong. Please try again later", 'error')
                    return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
            else:
                # Redirect to home page and show the error message
                flash("Something went wrong. Please try again later", 'error')
                return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
        else:
            # Redirect to home page and show the error message
            flash("Authentication failed: Invalid login credentials (Please try again)", 'error')
            return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)

    else:
        # Refresh page if it is not a post request
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/logout', methods=['POST', 'GET'])
def logout():
    "Logout client and stop docker container"

    # Stop the Docker container
    response = stopDocker()
    if response['status'] == 200:
         # Redirect to home page
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)
    else:
        flash("Session has expired, please relogin", 'error')
        return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/', methods=['POST', 'GET'])
def index():
    "Render sign in page"
    return render_template('index.html', major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@app.route('/reset', methods=['POST', 'GET'])
def reset():
    "Render reset password request page"
    return render_template("reset.html", major_release=major_release, minor_release=minor_release, patch_number=patch_number, build_number=build_number)


@socketio.on('disconnect')
def handle_disconnect():
    # Stop the docker as socket is disconnected
    stopDocker()

getVersion()
setAllowedPorts()

if __name__ == "__main__":

    # Delete existing redis data
    client.flush()

    # Close all existing docker containers
    stopAllDockers()

    # Get version data from version file
    getVersion()

    # Set allowed ports from settings file
    setAllowedPorts()

    # Set up data logging
    logging_handler()

    # Get port from environment variable
    port = os.environ.get('PORT')

    # Run the flask application
    socketio.run(app, debug=False, port=port, host='0.0.0.0', ssl_context=ssl_context, allow_unsafe_werkzeug=True)