import os, redis, functools

# Fetch the environment variables to access the redis instance
redis_host = os.environ.get('REDIS_HOST', '127.0.0.1')
redis_port = os.environ.get('REDIS_PORT', '6379')
redis_password = os.environ.get('REDIS_PASSWORD', None)

# Connect to the redis instance
redis_client = redis.Redis(host = redis_host, port = redis_port, password = redis_password, retry_on_timeout = True)


# Create a wrapper function for error handling
def handle_redis_error(fn):
    @functools.wraps(fn)
    def wrapper(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except Exception as err:
            print(f"Redis error: {err}")
    return wrapper


@handle_redis_error
def get(key):
    return redis_client.get(key)

@handle_redis_error
def set(key, value):
    return redis_client.set(key, value)

@handle_redis_error
def lrange(key, start, end):
    return redis_client.lrange(key, start, end)

@handle_redis_error
def append(key, value):
    return redis_client.append(key, value)

@handle_redis_error
def incr(key):
    return redis_client.incr(key)

@handle_redis_error
def incrby(key, amount):
    return redis_client.incrby(key, amount)

@handle_redis_error
def decr(key):
    return redis_client.decr(key)

@handle_redis_error
def decrby(key, amount):
    return redis_client.decrby(key, amount)

@handle_redis_error
def delete(key):
    return redis_client.delete(key)

@handle_redis_error
def keys(pattern):
    return redis_client.keys(pattern)

@handle_redis_error
def expire(key, seconds):
    return redis_client.expire(key, seconds)

@handle_redis_error
def ttl(key):
    return redis_client.ttl(key)

@handle_redis_error
def strlen(key):
    return redis_client.strlen(key)

@handle_redis_error
def publish(channel, message):
    return redis_client.publish(channel, message)

@handle_redis_error
def zadd(key, **kwargs):
    return redis_client.zadd(key, **kwargs)

@handle_redis_error
def zrange(key, start, end, withscores=False):
    return redis_client.zrange(key, start, end, withscores=withscores)

@handle_redis_error
def exists(key):
    return redis_client.exists(key)

@handle_redis_error
def lpush(key, *values):
    return redis_client.lpush(key, *values)

@handle_redis_error
def sadd(key, *members):
    return redis_client.sadd(key, *members)

@handle_redis_error
def sismember(key, member):
    return redis_client.sismember(key, member)

@handle_redis_error
def srem(key, *members):
    return redis_client.srem(key, *members)

@handle_redis_error
def smembers(key):
    return redis_client.smembers(key)

@handle_redis_error
def sinter(*keys):
    return redis_client.sinter(*keys)

@handle_redis_error
def sdiff(*keys):
    return redis_client.sdiff(*keys)

@handle_redis_error
def hset(name, key, value):
    return redis_client.hset(name, key, value)

@handle_redis_error
def hget(name, key):
    return redis_client.hget(name, key)

@handle_redis_error
def hdel(name, key):
    return redis_client.hdel(name, key)

@handle_redis_error
def hgetall(name):
    return redis_client.hgetall(name)

@handle_redis_error
def setex(key, expiry, value):
    return redis_client.setex(key, expiry, value)

@handle_redis_error
def flush():
    return redis_client.flushall()