const {Op} = require('sequelize');
const models = require('../models');
const httpErrors = require('http-errors');
const adminPortalURL = `${process.env.AP_SERVER_IP}:${process.env.AP_SERVER_PORT}`;
const axios = require('axios');
const session = require('./helpers/verifySession');

const createEntitlement = async (req, res, next) => {
    
    const transaction = await models.sequelize.transaction();
    const { userInfo } = req.userData;

    try{

        if(Object.keys(req.body).length === 0){
            return res.status(403).send({
                success: false,
                message: `Invalid parameters`
            });
        }


        const{username, hostname, ipv4} = req.body;

        const user = await models.LocalUsers.findOne({
            where: {
                userName: username
            }
        });

        if(!user){
            throw new httpErrors.NotFound(`No user found`);
        }

        const host = await models.DeviceInventories.findOne({
            where: {
                host_name: hostname
            }
        })

        if(!host){
            throw new httpErrors.NotFound(`No host found`);
        }

        const entitlement = await models.Entitlements.findOne({
            where: {
                name: mergeUserAndHostNames(username,hostname)
            }
        })

        if(entitlement){
            return res.status(409).send({
                success: false,
                message: `Entitlement already exists`
            });
        }

        const myServices = await models.Service.findAll({
            where: {
                ipv4_address: ipv4
            }
        });


        const client = await models.Client.findOne({
            where: {
                user_id: user.id
            }
        });

        if(!user){
            throw new httpErrors.NotFound(`No user found`);
        }

        if(!host){
            throw new httpErrors.NotFound(`No host found`);
        }

        if(myServices.length === 0 || !myServices){
            throw new httpErrors.NotFound(`No hostname/ip found`);
        }

        var entitlementName = mergeUserAndHostNames(username,hostname);
    


        const Entitlement = await models.sequelize.query(
            "INSERT INTO Entitlements (name, policy_type, createdAt, updatedAt) VALUES (?, ?, NOW(), NOW())",
            {
              replacements: [entitlementName, 'User Agent'],
              type: models.sequelize.QueryTypes.INSERT,
              transaction
            }
        );


        const entitlementId = Entitlement[0];


        for(const service of myServices){
            const test1 = await models.ServicePolicies.create({
                service_id: service.id,
                policy_type_id: 1,
                policy_id: entitlementId
              }, {transaction});
        
        }


          

        await models.ClientPolicies.create({
            client_id: client.id,
            policy_type_id: 1,
            policy_id: entitlementId
        }, {transaction});


        const logMessage = {
            action: `Created entitlement ${entitlementName}`,
        };


        await models.AdminAccessLogs.create(
            {
                userId: userInfo,
                action: logMessage.action,
                priority: 'Medium',
                requestBody: JSON.stringify(req.body),
                requestParams: JSON.stringify(req.params),
                requestQueryParams: JSON.stringify(req.query),
                createdAt: new Date(),
                updatedAt: new Date()
            },
            { transaction }
        );
        


        await transaction.commit();


          res.status(200).send({
            success: true,
            message: `Entitlement created successfully`
        });

        
    }catch(e){
        await transaction.rollback();
        if (e instanceof httpErrors.NotFound) {
            return res.status(404).send({
                success: false,
                message: e.message
            });
        }else{
            return res.status(500).send({
                success: false,
                message: `An error occured`
            });
        }
    }
}


const deleteEntitlement = async (req, res, next) => {

    const transaction = await models.sequelize.transaction();
    const { userInfo } = req.userData;

    try{

        if(Object.keys(req.body).length === 0){
            return res.status(403).send({
                success: false,
                message: `Invalid parameters`
            });
        }


        const {username, hostname} = req.body;

        const user = await models.LocalUsers.findOne({
            where: {
                userName: username
            }
        });

        if(!user){
            throw new httpErrors.NotFound(`No user found`);
        }

        const host = await models.DeviceInventories.findOne({
            where: {
                host_name: hostname
            }
        })

        if(!host){
            throw new httpErrors.NotFound(`No host found`);
        }

        var entitlementName = mergeUserAndHostNames(username,hostname);

        const entitlement = await models.Entitlements.findOne({
            where: {
                name: entitlementName,
            }
          });

        if (!entitlement) {
            throw new httpErrors.NotFound(`Entitlement not found`);
        }


        const logMessage = {
            action: `Deleted entitlement ${entitlementName}`,
        };


        await models.AdminAccessLogs.create(
            {
                userId: userInfo,
                action: logMessage.action,
                priority: 'Medium',
                requestBody: JSON.stringify(req.body),
                requestParams: JSON.stringify(req.params),
                requestQueryParams: JSON.stringify(req.query),
                createdAt: new Date(),
                updatedAt: new Date()
            },
            { transaction }
        );




        await entitlement.destroy({ transaction });
        await transaction.commit();



        res.status(200).send({
            success: true,
            message: `Entitlement deleted successfully`
        });


    }catch(e){
        await transaction.rollback();

        if (e instanceof httpErrors.NotFound) {
            return res.status(404).send({
                success: false,
                message: e.message
            });
        }else{
            return res.status(500).send({
                success: false,
                message: `An error occured`
            });
        }
    }
}



const mergeUserAndHostNames = (username,hostname) => {
    return '{'+username+'}@{'+hostname+'}';
  }





module.exports = {
    createEntitlement,
    deleteEntitlement
}