const models = require('../models');
const language = require('../routes/common/language');
const httpErrors = require('http-errors');

/* GET idps listing. */
const getIdps = async (req, res, next) => {
  try {
    const {
      limit,
      offset,
      sort='createdAt',
      dir='desc',
      action,
      searchKey=''
    } = req.query;

    const options = {
      where: { },
      limit: limit,
      offset: offset,
      order: []
    }; 

    if(searchKey.trim()){
      options.where.name = {
        [Op.like]: `%${searchKey.trim()}%`
      }
    }

    if (action === 'active') {
      options.where.is_active = 1;
    }

    const allowedSortValues = [
      'name',
      'id',
      'createdAt'
    ];

    if (allowedSortValues.includes(sort)) {
      options.order.push([models.sequelize.literal(sort), dir]);
    }

    const result = await models.Idp.paginate(options);
    res.json(result);
  } catch (error) {
    next(error);
  }
}


//check whether idp already exist or not
const checkIsIdpUniqueOrNot = async (idp, update) => {
  const { 
    name, 
    sp_id, 
    assertion_url, 
    id 
  } = idp;

  const whereClause = {
    $or: [
      {
        sp_id: {
          $eq: sp_id
        }
      },
      {
        name: {
          $eq: name
        }
      },
      {
        assertion_url: {
          $eq: assertion_url
        }
      }
    ]
  };

  if (update) {
    whereClause.id = {
      $ne: id
    };
  }

  try {
    const count = await models.Idp.count({
      where: whereClause
    });

    if (count >= 1) {
      return false;
    }

    return true;
  } catch (error) {
    return false;
  }
};


/* GET update idp. */
const udpateIdp = async (req, res, next) => {
  try {
    
    const { idp } = req.body;

    const count = await checkIsIdpUniqueOrNot(idp, true);
    if (!count) {
      throw new httpErrors.BadRequest('Duplicate entry already exists.');
    }

    const idpToUpdate = await models.Idp.findOne({
      where: {
        id: req.params.id
      }
    })
    
    if (!idpToUpdate) {
      throw new httpErrors.NotFound("No idp to update.");
    }

    const updatedIdp = await idpToUpdate.update(idp); 

    if(!updatedIdp){
      throw new httpErrors.BadGateway("Failed to update idp.");
    }

    res.json({
      success: true,
      message: 'Edited Successfully'
    });

  } catch (error) {
    next(error);
  }
}


/* Get a single idp detail */
const getIdp = async (req, res, next) => {
  try {
    
    const options = {
      where: {
        id: req.params.id
      }
    };

    const idp = await models.Idp.findAll(options)
    
    res.json(idp);
  
  } catch (error) {
    next(error);
  }
}


// add new idp
const addIdp = async (req, res, next) => {
  try {

    const { idp } = req.body;

    const count = await checkIsIdpUniqueOrNot(idp, false);
    
    if (!count) {
      throw new httpErrors.Conflict('Duplicate entry already exists.');
    }

    const createdIdp = await models.Idp.create(idp);
    if(!createdIdp){
      throw new httpErrors.BadGateway("Failed to created idp.");
    }

    res.json({
      success: true,
      message: language.get['ADD_SUCCESS']
    }); 

  } catch (error) {
    next(error);
  }
}


// delete a single idp
const deleteIdp = async (req, res, next) => {
  try {

    const id = req.params.id;
    const idp = await models.Idp.findOne({
      where: {
        id: id
      }
    })

    if(!idp){
      throw new httpErrors.NotFound("No idp to delete.");
    }

    const updatedIdp = await idp.destroy({ });
    if(!updatedIdp){
      throw new httpErrors.BadGateway("Failed to delete idp.");
    }

    res.json({
      success: true,
      message: language.get['REMOVE_SUCCESS']
    });

  } catch (error) {
    next(error);
  }
}

module.exports = {
  deleteIdp,
  addIdp,
  getIdp,
  udpateIdp,
  getIdps
}
