const models = require('../models');
const httpErrors = require('http-errors');
const {Op} = require('sequelize');
const { publishClients, actionType } = require('../routes/common/event-publisher');


const updateUserRiskScore = async (req, res, next) => {
    const { risk_score, user_id } = req.body;

    try {
        if (!risk_score || !user_id) {
            return res.status(400).send({
                success: false,
                message: 'Both risk_score and user_id are required parameters.',
            });
        }

        const user = await models.LocalUsers.findOne({
            where: {
                [models.Sequelize.Op.or]: [
                    { userName: user_id },
                    { email: user_id },
                ],
            },
        });

        if (!user) {
            throw new httpErrors.NotFound(`User not found`);
        }

        const clients = await models.Client.findAll({
            where: {
                user_id: user.id,
            },
        });

        if (!clients || clients.length === 0) {
            throw new httpErrors.NotFound(`No clients found for the user`);
        }

        await models.Client.update(
            { risk_score: risk_score },
            {
                where: {
                    user_id: user.id,
                },
            }
        );

        const clientIds = clients.map(client => client?.id);

        await publishClients(actionType.UPDATED, {
            ids: clientIds,
            mode: 0,
            data: { risk_score: +risk_score },
        });

        res.status(200).send({
            success: true,
            message: `Risk score updated successfully for all clients of user ${user_id}.`,
        });
    } catch (error) {
        if (error instanceof httpErrors.NotFound) {
            return res.status(404).send({
                success: false,
                message: error.message,
            });
        } else {
            return res.status(500).send({
                success: false,
                message: `An error occurred`,
            });
        }
    }
};

const updateDeviceRiskScore = async (req, res, next) => {
    const { risk_score, device_id } = req.body;

    try {
        if (!risk_score || !device_id) {
            return res.status(400).send({
                success: false,
                message: 'Both risk_score and device_id are required parameters.',
            });
        }

        const device = await models.DeviceInventories.findOne({
            where: {
                device_id: device_id,
            },
        });

        if (!device) {
            throw new httpErrors.NotFound(`Device not found`);
        }

        const client = await models.Client.findOne({
            where: {
                device_id: device.id,
            },
        });

        if (!client) {
            throw new httpErrors.NotFound(`Client not found for the device`);
        }

        await models.Client.update(
            { risk_score: risk_score },
            {
                where: {
                    device_id: device.id,
                },
            }
        );

        await publishClients(actionType.UPDATED, {
            ids: [ client?.id ],
            mode: 0,
            data: { risk_score: +risk_score },
            type: client?.client_type
        });

        res.status(200).send({
            success: true,
            message: `Risk score updated successfully for client ${client.Name}.`,
        });
    } catch (error) {
        if (error instanceof httpErrors.NotFound) {
            return res.status(404).send({
                success: false,
                message: error.message,
            });
        } else {
            return res.status(500).send({
                success: false,
                message: `An error occurred`,
            });
        }
    }
};

module.exports = {
    updateUserRiskScore,
    updateDeviceRiskScore
}