const httpErrors = require('http-errors');
const _ = require('lodash');

('use strict');
module.exports = (sequelize, DataTypes) => {
  const Entitlements = sequelize.define(
    'Entitlements',
    {
      id: {
        allowNull: false,
        autoIncrement: true,
        primaryKey: true,
        type: DataTypes.INTEGER
      },
      name: {
        type: DataTypes.STRING
      },
      access_restriction: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      access_start_time: {
        type: DataTypes.TIME
      },
      access_end_time: {
        type: DataTypes.TIME
      },
      months: {
        type: DataTypes.STRING(50),
        get() {
          const rawValue = this.getDataValue('months');
          return rawValue ? rawValue.split(',').map(Number) : null;
        }
      },
      days: {
        type: DataTypes.STRING(50),
        get() {
          const rawValue = this.getDataValue('days');
          return rawValue ? rawValue.split(',').map(Number) : null;
        }
      },
      is_enabled: {
        type: DataTypes.BOOLEAN,
        defaultValue: true
      },
      all_clients: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      all_services: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      policy_type: {
        type: DataTypes.STRING
      },
      createdAt: {
        allowNull: false,
        type: DataTypes.DATE
      },
      updatedAt: {
        allowNull: false,
        type: DataTypes.DATE
      }
    },
    {}
  );
  Entitlements.associate = function(models) {
    // associations can be defined here
  };

  // Entitlements.addHook('beforeCreate', checkIfPolicyExist);
  // Entitlements.addHook('beforeUpdate', checkIfPolicyExist);

  /**
   * Check if the name of the policy already exist
   * @param {string} policy
   */
  async function checkIfPolicyExist(policy, options) {
    const models = require('../models');
    const isUpdate = options.updateData ? true : false;


    const all_clients = isUpdate ? options.updateData.all_clients : policy.all_clients; 
    const all_services = isUpdate ? options.updateData.all_services : policy.all_services;

    policy.dataValues.name = _.upperFirst(
      _.toLower(policy.dataValues.name.trim())
    );
    if (!policy.dataValues.name)
      throw new httpErrors.UnprocessableEntity('Name is required');

    const name = policy.dataValues.name;
    if (
      options.fields.includes('name') &&
      policy.name !== policy.previous('name')
    ) {
      const result = await Entitlements.findOne({
        where: {
          name
        }
      });

      if (result)
        throw new httpErrors.Conflict('Policy with this name already exists');
    }

    if(all_services && all_clients){
      const conflicting_policy = await Entitlements.findOne({
        where: {
          all_clients: true,
          all_services: true,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
  
      if(conflicting_policy){
        throw new httpErrors.Conflict(`Policy already exists with same configuration. Existing policy name is ${conflicting_policy.name}.`);
      }
    }

    if(all_services && !all_clients){
      const clientIds = isUpdate ? options.updateData.client : policy.CPolicies.map(_client => _client.client_id);
      const clientGroupIds = isUpdate ? options.updateData.clientGroup : policy.CGroupPolicies.map(_policy => _policy.client_group_id);

      const [conflicting_clients, conflicting_client_groups] = await Promise.all([
        models.ClientPolicies.findAll({
          attributes: [
            'policy_id',
            'client_id'
          ],
          where: {
            client_id:{
              [Op.in]: clientIds
            },
            policy_type_id: 1,
            policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.ClientGroupPolicies.findAll({
          attributes: [
            'policy_id',
            'client_group_id'
          ],
          where: {
            client_group_id:{
              [Op.in]: clientGroupIds
            },
            policy_id: {
              [Op.ne]: policy.id
            },
            policy_type_id: 1
          },
          raw: true
        })
      ])

      if(conflicting_clients.length > 0 || conflicting_client_groups.length > 0){

        const policyIds = [
          ...conflicting_clients.map(_c => _c.policy_id), 
          ...conflicting_client_groups.map(_cg => _cg.policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await Entitlements.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }else{
        const AccessPolicyWithAllClients = await Entitlements.findAll({
          where: {
            all_clients: true,
            all_services: false
          }
        });
  
        if(AccessPolicyWithAllClients.length){
          const names = AccessPolicyWithAllClients.map(_p => _p.name);
          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }
    }

    if(!all_services && all_clients){
      const serviceIds = isUpdate ? options.updateData.service : policy.SPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupPolicies.map(_policy => _policy.service_group_id);
      
      const [services, serviceGroups] = await Promise.all([
        models.ServicePolicies.findAll({
          attributes: [
            'policy_id',
            'service_id'
          ],
          where: {
            service_id: {
              [Op.in]: serviceIds
            },
            policy_id: {
              [Op.ne]: policy.id
            },
            policy_type_id: 1
          },
          raw: true
        }),
        models.ServiceGroupPolicies.findAll(
          {
            attributes: [
              'policy_id',
              'service_group_id'
            ],
            where: {
              service_group_id: {
                [Op.in]: serviceGroupIds
              },
              policy_id: {
                [Op.ne]: policy.id
              },
              policy_type_id: 1
            },
            raw: true
          }
        )
      ]);

      if(services.length > 0 || serviceGroups.length > 0){
        const policyIds = [
          ...services.map(_c => _c.policy_id), 
          ...serviceGroups.map(_cg => _cg.policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await Entitlements.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

      const allClientPolicy = await models.Entitlements.findAll({
        where: {
          all_services: true,
          all_clients: false,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
      
      if(allClientPolicy.length > 0){
        const names = allClientPolicy.map(_p => _p.name);
        throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
      }
    
    }

    if(!all_services && !all_clients){
      const clientIds = isUpdate ? options.updateData.client : policy.CPolicies.map(_client => _client.client_id);
      const clientGroupIds = isUpdate ? options.updateData.clientGroup : policy.CGroupPolicies.map(_policy => _policy.client_group_id);
      const serviceIds = isUpdate ? options.updateData.service :  policy.SPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupPolicies.map(_policy => _policy.service_group_id);

      const [clients, clientGroups] = await Promise.all([
        models.ClientPolicies.findAll({
            attributes: [
              'policy_id',
              'client_id'
            ],
            where: {
              client_id: {
                [Op.in]: clientIds
              },
              policy_type_id: 1,
              policy_id: {
                [Op.ne]: policy.id
              }
            },
            raw: true
        }),
        models.ClientGroupPolicies.findAll({
          attributes: [
            'policy_id',
            'client_group_id'
          ],
          where: {
            client_group_id: {
              [Op.in]: clientGroupIds
            },
            policy_id: {
              [Op.ne]: policy.id
            },
            policy_type_id: 1
          },
          raw: true
        })
      ]); 
      
      const policyIds = clients.map(client => {
        return client.policy_id;
      });

      clientGroups.forEach(group => {
        policyIds.push(group.policy_id);
      })
      
      const [services, serviceGroups] = await Promise.all([
            models.ServicePolicies.findAll({
              attributes: [
                'policy_id',
                'service_id'
              ],
              where: {
                service_id: {
                  [Op.in]: serviceIds
                },
                policy_id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                },
                policy_type_id: 1
              },
              raw: true
            }),
            models.ServiceGroupPolicies.findAll(
              {
                attributes: [
                  'policy_id',
                  'service_group_id'
                ],
                where: {
                  service_group_id: {
                    [Op.in]: serviceGroupIds
                  },
                  policy_id: {
                    [Op.in]: policyIds,
                    [Op.ne]: policy.id
                  },
                  policy_type_id: 1
                },
                raw: true
              }
            )
          ]);
    
        if(services.length > 0 || serviceGroups.length > 0){
          const policyIds = [
            ...services.map(_c => _c.policy_id), 
            ...serviceGroups.map(_cg => _cg.policy_id)
          ];
  
          if(policyIds.length > 0){
            const conflicting_policies = await Entitlements.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policyIds
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
  
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    
        if(policyIds.length){
            const allServicePolicy = await models.Entitlements.findAll({
              where: {
                id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                },
                all_services: true,
                all_clients: false
              }
            });
            
            if(allServicePolicy.length > 0){
              const names = allServicePolicy.map(_p => _p.name);
              throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
            }
        }

        const allClientPolicy = await models.Entitlements.findAll({
          where: {
            all_services: false,
            all_clients: true,
            id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        });

        _policyIds = allClientPolicy.map(_policy => _policy.id);

        const [_services, _serviceGroups] = await Promise.all([
          models.ServicePolicies.findAll({
            attributes: [
              'policy_id',
              'service_id'
            ],
            where: {
              service_id: {
                [Op.in]: serviceIds
              },
              policy_id: {
                [Op.in]: _policyIds,
                [Op.ne]: policy.id
              },
              policy_type_id: 1
            },
            raw: true
          }),
          models.ServiceGroupPolicies.findAll(
            {
              attributes: [
                'policy_id',
                'service_group_id'
              ],
              where: {
                service_group_id: {
                  [Op.in]: serviceGroupIds
                },
                policy_id: {
                  [Op.in]: _policyIds,
                  [Op.ne]: policy.id
                },
                policy_type_id: 1
              },
              raw: true
            }
          )
        ]);
        
        if(_services.length > 0 || _serviceGroups.length > 0){
          const policy_ids = [
            ..._services.map(_c => _c.policy_id), 
            ..._serviceGroups.map(_cg => _cg.policy_id)
          ];
  
          if(policy_ids.length > 0){
            const conflicting_policies = await Entitlements.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policy_ids
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    }

    if (!policy.dataValues.access_restriction) {
      policy.dataValues.days = null;
      policy.dataValues.months = null;
      policy.dataValues.access_start_time = null;
      policy.dataValues.access_end_time = null;
    }
  }

  return Entitlements;
};
