'use strict';
const moment = require('moment');
const { authenticator } = require('otplib');
const qrcode = require('qrcode');
const { promisify } = require('util');
const toDataURL = promisify(qrcode.toDataURL);
const { where, Op, fn, col, literal } = require('sequelize');

authenticator.options = {
  window: 1
}

module.exports = function(sequelize, DataTypes) {
  var Client = sequelize.define('Client', {
    Name: DataTypes.STRING,
    Email: DataTypes.STRING,
    agent_id: DataTypes.INTEGER,
    wrong_attempts:DataTypes.INTEGER,
    is_active: DataTypes.BOOLEAN,
    ipv4_address: DataTypes.STRING,
    ENC_KEY_LEN: DataTypes.INTEGER,
    HMAC_KEY_LEN: DataTypes.INTEGER,
    access_restrictions: DataTypes.BOOLEAN,
    user: DataTypes.STRING,
    enc_key: DataTypes.STRING,
    HMacKey: DataTypes.STRING,
    device_os: DataTypes.STRING,
    device_hardware_id: DataTypes.STRING,
    cacert: DataTypes.BLOB('tiny'),
    clientcrt: DataTypes.BLOB('tiny'),
    clientpem: DataTypes.BLOB('tiny'),
    link_key: DataTypes.STRING,
    counter: DataTypes.STRING,
    installer_logs: DataTypes.TEXT('long'),
    link_key_expiry: DataTypes.DATE,
    heart_beat: DataTypes.BOOLEAN,
    heart_beat_sent_at: DataTypes.DATE,
    installer_status: DataTypes.STRING,
    cacer: DataTypes.BLOB('tiny'),
    clientcer: DataTypes.BLOB('tiny'),
    clientp12: DataTypes.BLOB('tiny'),
    isDeviceIdSet: DataTypes.BOOLEAN,
    package_sent_at: DataTypes.DATE,
    client_darkening: DataTypes.BOOLEAN,
    geoip_track: DataTypes.BOOLEAN,
    package_used: DataTypes.BOOLEAN,
    qr_json: DataTypes.TEXT('long'),
    onPrenDownloadLink: DataTypes.STRING,
    isDownloadPackage: DataTypes.BOOLEAN,
    packageDownloaded: DataTypes.BOOLEAN,
    client_type: DataTypes.STRING,
    isCancelled: DataTypes.BOOLEAN,
    is_ldap_deleted: DataTypes.BOOLEAN,
    is_local_deleted: DataTypes.BOOLEAN,
    client_status: DataTypes.INTEGER,
    client_mfa: DataTypes.BOOLEAN,
    device_os_full_name: DataTypes.STRING,
    domain_name: DataTypes.STRING,
    antivirus_installed: DataTypes.STRING,
    serial_number: DataTypes.STRING,
    linOTP_token: DataTypes.JSON,
    OTP_image_data: DataTypes.TEXT('long'),
    is_local: DataTypes.BOOLEAN,
    client_av: DataTypes.STRING,
    client_os_version: DataTypes.STRING,
    waiting_reason: DataTypes.STRING,
    recieved_os_version: DataTypes.STRING,
    client_version: DataTypes.STRING,
    recieved_antivirus: DataTypes.STRING,
    biometric_state: DataTypes.BOOLEAN,
    client_epp: DataTypes.BOOLEAN,
    recieved_epp: DataTypes.STRING,
    mfa_status: DataTypes.STRING,
    auto_updates: DataTypes.BOOLEAN,
    user_id: DataTypes.INTEGER,
    device_id: DataTypes.INTEGER,
    shared_secret: DataTypes.STRING,
    risk_score: {
      type: DataTypes.INTEGER,
      allowNull: false,
      defaultValue: 1
    },
    event: {
      type: DataTypes.BOOLEAN,
      allowNull: true,
    },
    skip_otp_check: {
      type: DataTypes.VIRTUAL,
      defaultValue: false
    },
    is_online: {
      type: new DataTypes.VIRTUAL(DataTypes.BOOLEAN, [
        'heart_beat',
        'heart_beat_sent_at'
      ]),
      get: function() {
        const isHeartBeat = this.get('heart_beat');
        if (!isHeartBeat) return false;
        let allowedTimeInUtc = moment().subtract(120, 'seconds');
        const diffInSeconds = moment(this.get('heart_beat_sent_at'))
          .utc()
          .diff(allowedTimeInUtc, 'second');
        return diffInSeconds >= 0 ? true : false;
      }
    },
    is_strong_mfa: {
      type: DataTypes.BOOLEAN,
      defaultValue: 0
    }
  });



  //client instance level methods

  /**
   * @description check whether client is bound to a specific Server Bridge
   * @return {Boolean}
   */

  Client.prototype.isBound = async function(){

    const model = require('../models');

    const clientId = this.get('id');

    try{

      const count = await model.ServerBridge.count({
        where:{
          clientId: clientId
        }
      });

      if(count) return true;

      return false;

    }catch(error){
      return false;
    }

  }

  Client.addScope('boundServerBridge', (attributes = ['name']) => {
    const models = require('../models');
    return {
      include:[
        { 
          model: models.ServerBridge,
          attributes
        }
      ]
    };
  });
  
  Client.addScope('online', isOnline => {
    return {
      where: {
        [isOnline ? Op.and : Op.or]: [
          {
            heart_beat: isOnline
              ? 1
              : {
                  [Op.in]: [0]
                }
          },
          where(
            fn(
              'TIMESTAMPDIFF',
              literal('SECOND'),
              fn('DATE_SUB', fn('NOW'), literal('INTERVAL 5 MINUTE')),
              col('heart_beat_sent_at')
            ),
            {
              [isOnline ? Op.gte : Op.lte]: 2
            }
          )
        ]
      }
    };
  });
  
  Client.addHook('beforeCreate', generateOtp);

  Client.prototype.generateOtp = async function() {
    await generateOtp(this);
    return this.save();
  }

  Client.prototype.verifyOtp = function(token) {
    if (!this.linOTP_token) return false;
    const check = authenticator.checkDelta(token, this.linOTP_token);
    return Number.isInteger(check);
  }

  Client.associate = function(models) {
    Client.belongsToMany(models.Service, {
      through: 'ClientService',
      foreignKey: 'clientId'
    });
    Client.belongsToMany(models.Group, {
      through: 'GroupClient',
      foreignKey: 'clientId'
    });
    Client.belongsToMany(models.Entitlements, {
      through: models.ClientPolicies,
      foreignKey: 'client_id'
    });
    Client.belongsToMany(models.DevicePolicies, {
      through: models.ClientDevicePolicies,
      foreignKey: 'client_id'
    });
    Client.hasMany(models.GroupClient, { foreignKey: 'clientId' });
    Client.belongsToMany(models.SoftwarePolicies, { 
      through: models.ClientSoftwarePolicies,
      foreignKey: 'client_id' 
    });

    Client.hasMany(models.ClientPolicies, { foreignKey: 'client_id' });
    Client.hasMany(models.ServerBridgeDeviceId, { foreignKey: "client_id"});


    Client.hasMany(models.ClientAccessRestriction, { foreignKey: 'clientId' });
    Client.hasMany(models.SaasBridge, { foreignKey: 'clientId' });
    Client.hasMany(models.Connection, { foreignKey: 'AGENT_ID' });
    Client.hasMany(models.GlobalIps, { foreignKey: 'clientId' });
    Client.hasMany(models.ServiceAccessLogs, { foreignKey: 'agent_id' });
    Client.hasOne(models.ServerBridge,{foreignKey: 'clientId'});
    Client.belongsTo(models.DeviceInventories, { foreignKey: 'device_id' });
    Client.belongsTo(models.LocalUsers, { foreignKey: 'user_id' });

  };

  async function generateOtp(user, options) {
    let token = null;

    if(!user.skip_otp_check) {
      token = await Client.findOne({
        attributes: ['linOTP_token', 'OTP_image_data', 'user'],
        where: {
          user: user.user,
          linOTP_token: {
            [Op.ne]: null
          }
        }
      });
    }

    if(token) {
      user.OTP_image_data = token.OTP_image_data;
      user.linOTP_token = token.linOTP_token;
    } else {
      user.linOTP_token = authenticator.generateSecret();
      const otpauth = authenticator.keyuri(user.user, 'INVISILY', user.linOTP_token);
      const img = await toDataURL(otpauth);
      user.OTP_image_data = img.split(';base64,').pop().slice(0, -5);
    }
  }

  return Client;
};
