const httpErrors = require('http-errors');
const _ = require('lodash');

('use strict');
module.exports = (sequelize, DataTypes) => {
  const DevicePolicies = sequelize.define(
    'DevicePolicies',
    {
      id: {
        allowNull: false,
        autoIncrement: true,
        primaryKey: true,
        type: DataTypes.INTEGER
      },

      name: {
        type: DataTypes.STRING
      },
      
      is_enabled: {
        type: DataTypes.BOOLEAN
      },
      
      antivirus: {
        type: DataTypes.STRING
      },
      is_epp_client: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      is_blocked: {
        type: DataTypes.BOOLEAN,
        defaultValue: true
      },
      all_clients: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      all_services: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      risk_score_threshold: {
        type: DataTypes.INTEGER,
        allowNull: false,
        defaultValue: 1
      },
      createdAt: {
        allowNull: false,
        type: DataTypes.DATE
      },
      updatedAt: {
        allowNull: false,
        type: DataTypes.DATE
      }
    },
    {}
  );
  DevicePolicies.associate = function(models) {
    // associations can be defined here
    DevicePolicies.hasMany(models.ClientDevicePolicies, {
      foreignKey: 'device_policy_id',
      as: 'CDPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });
    DevicePolicies.hasMany(models.ServiceDevicePolicies, {
      foreignKey: 'device_policy_id',
      as: 'SDPolicies',
      onDelete: 'CASCADE'
    });
    DevicePolicies.hasMany(models.ClientGroupDevicePolicies, {
      foreignKey: 'device_policy_id',
      as: 'CGroupDPolicies',
      onDelete: 'CASCADE'
    });
    DevicePolicies.hasMany(models.ServiceGroupDevicePolicies, {
      foreignKey: 'device_policy_id',
      as: 'SGroupDPolicies',
      onDelete: 'CASCADE'
    });

    DevicePolicies.belongsToMany(models.Client, {
      foreignKey: 'device_policy_id',
      through: models.ClientDevicePolicies
    });
    DevicePolicies.belongsToMany(models.Service, {
      foreignKey: 'device_policy_id',
      through: models.ServiceDevicePolicies
    });
    DevicePolicies.belongsToMany(models.Group, {
      foreignKey: 'device_policy_id',
      through: models.ClientGroupDevicePolicies
    });
    DevicePolicies.belongsToMany(models.ServiceGroup, {
      foreignKey: 'device_policy_id',
      through: models.ServiceGroupDevicePolicies
    });
  };

  DevicePolicies.addScope('getEnablePolicy', {
    where: {
      is_enabled: 1
    }
  });

  //Prototypes

  //Hooks

  DevicePolicies.addHook('beforeCreate', checkIfPolicyExist);
  DevicePolicies.addHook('beforeUpdate', checkIfPolicyExist);


  // DevicePolicies.addHook('afterDestroy', updateClients);
  // DevicePolicies.addHook('afterUpdate', async (policy, options) => {
  //   if (!policy.dataValues.is_enabled) {
  //     await updateClients(policy, options);
  //   }
  // });

  /**
   *
   * @param {Object} policy
   */
  // async function updateClients(policy, options) {
  //   const { transaction } = options;
  //   const { os } = policy.dataValues;
  //   const models = require('../models');
  //   try {
  //     await models.Client.update(
  //       { isPolicyBlocked: true },
  //       { where: { device_os: os }, transaction }
  //     );
  //     await transaction.commit();
  //   } catch (err) {
  //     await transaction.rollback();
  //     throw new Error('Could not update client');
  //   }
  // }

  /**
   * Check if the name of the policy already exist
   * @param {string} policy
   */
  async function checkIfPolicyExist(policy, options) {

    const models = require('../models');
    const isUpdate = options.updateData ? true : false;


    const all_clients = isUpdate ? options.updateData.all_clients : policy.all_clients; 
    const all_services = isUpdate ? options.updateData.all_services : policy.all_services;

    policy.dataValues.name = _.upperFirst(
      _.toLower(policy.dataValues.name.trim())
    );
    if (!policy.dataValues.name)
      throw new httpErrors.UnprocessableEntity('Name is required');

    const name = policy.dataValues.name;
    if (
      options.fields.includes('name') &&
      policy.name !== policy.previous('name')
    ) {
      const result = await DevicePolicies.findOne({
        where: {
          name
        }
      });

      if (result)
        throw new httpErrors.Conflict('Policy with this name already exist');
    }



    if(all_services && all_clients){
      const conflicting_policy = await DevicePolicies.findOne({
        where: {
          all_clients: true,
          all_services: true,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
  
      if(conflicting_policy){
        throw new httpErrors.Conflict(`Policy already exists with same configuration. Existing policy name is ${conflicting_policy.name}.`);
      }
    }

    if(all_services && !all_clients){
      const clientIds = isUpdate ? options.updateData.client : policy.CDPolicies.map(_client => _client.client_id);
      const clientGroupIds = isUpdate ? options.updateData.clientGroup : policy.CGroupDPolicies.map(_policy => _policy.client_group_id);

      const [conflicting_clients, conflicting_client_groups] = await Promise.all([
        models.ClientDevicePolicies.findAll({
          attributes: [
            'device_policy_id',
            'client_id'
          ],
          where: {
            client_id:{
              [Op.in]: clientIds
            },
            device_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.ClientGroupDevicePolicies.findAll({
          attributes: [
            'device_policy_id',
            'client_group_id'
          ],
          where: {
            client_group_id:{
              [Op.in]: clientGroupIds
            },
            device_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        })
      ])

      if(conflicting_clients.length > 0 || conflicting_client_groups.length > 0){
        const policyIds = [
          ...conflicting_clients.map(_c => _c.device_policy_id), 
          ...conflicting_client_groups.map(_cg => _cg.device_policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await DevicePolicies.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }else{
        const SecurityPolicyWithAllClients = await DevicePolicies.findAll({
          where: {
            all_clients: true,
            all_services: false
          }
        });
  
        if(SecurityPolicyWithAllClients.length){
          const names = SecurityPolicyWithAllClients.map(_p => _p.name);
          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

    }

    if(!all_services && all_clients){
      const serviceIds = isUpdate ? options.updateData.service : policy.SDPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupDPolicies.map(_policy => _policy.service_group_id);
      
      const [services, serviceGroups] = await Promise.all([
        models.ServiceDevicePolicies.findAll({
          attributes: [
            'device_policy_id',
            'service_id'
          ],
          where: {
            service_id: {
              [Op.in]: serviceIds
            },
            device_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.ServiceGroupDevicePolicies.findAll(
          {
            attributes: [
              'device_policy_id',
              'service_group_id'
            ],
            where: {
              service_group_id: {
                [Op.in]: serviceGroupIds
              },
              device_policy_id: {
                [Op.ne]: policy.id
              }
            },
            raw: true
          }
        )
      ]);

      if(services.length > 0 || serviceGroups.length > 0){
        const policyIds = [
          ...services.map(_c => _c.device_policy_id), 
          ...serviceGroups.map(_cg => _cg.device_policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await DevicePolicies.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

      const allClientPolicy = await models.DevicePolicies.findAll({
        where: {
          all_services: true,
          all_clients: false,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
      
      if(allClientPolicy.length > 0){
        const names = allClientPolicy.map(_p => _p.name);
        throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
      }
    
    }

    if(!all_services && !all_clients){
      const clientIds = isUpdate ? options.updateData.client : policy.CDPolicies.map(_client => _client.client_id);
      const clientGroupIds = isUpdate ? options.updateData.clientGroup : policy.CGroupDPolicies.map(_policy => _policy.client_group_id);
      const serviceIds = isUpdate ? options.updateData.service :  policy.SDPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupDPolicies.map(_policy => _policy.service_group_id);

      const [clients, clientGroups] = await Promise.all([
        models.ClientDevicePolicies.findAll({
            attributes: [
              'device_policy_id',
              'client_id'
            ],
            where: {
              client_id: {
                [Op.in]: clientIds
              },
              device_policy_id: {
                [Op.ne]: policy.id
              }
            },
            raw: true
        }),
        models.ClientGroupDevicePolicies.findAll({
          attributes: [
            'device_policy_id',
            'client_group_id'
          ],
          where: {
            client_group_id: {
              [Op.in]: clientGroupIds
            },
            device_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        })
      ]); 
      
      const policyIds = clients.map(client => {
        return client.device_policy_id;
      });

      clientGroups.forEach(group => {
        policyIds.push(group.device_policy_id);
      })
      
      const [services, serviceGroups] = await Promise.all([
            models.ServiceDevicePolicies.findAll({
              attributes: [
                'device_policy_id',
                'service_id'
              ],
              where: {
                service_id: {
                  [Op.in]: serviceIds
                },
                device_policy_id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                }
              },
              raw: true
            }),
            models.ServiceGroupDevicePolicies.findAll(
              {
                attributes: [
                  'device_policy_id',
                  'service_group_id'
                ],
                where: {
                  service_group_id: {
                    [Op.in]: serviceGroupIds
                  },
                  device_policy_id: {
                    [Op.in]: policyIds,
                    [Op.ne]: policy.id
                  }
                },
                raw: true
              }
            )
          ]);
    
        if(services.length > 0 || serviceGroups.length > 0){
          const policyIds = [
            ...services.map(_c => _c.device_policy_id), 
            ...serviceGroups.map(_cg => _cg.device_policy_id)
          ];
  
          if(policyIds.length > 0){
            const conflicting_policies = await DevicePolicies.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policyIds
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
  
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    
        if(policyIds.length){
            const allServicePolicy = await models.DevicePolicies.findAll({
              where: {
                id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                },
                all_services: true,
                all_clients: false
              }
            });
            
            if(allServicePolicy.length > 0){
              const names = allServicePolicy.map(_p => _p.name);
              throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
            }
        }

        const allClientPolicy = await models.DevicePolicies.findAll({
          where: {
            all_services: false,
            all_clients: true,
            id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        });

        _policyIds = allClientPolicy.map(_policy => _policy.id);

        const [_services, _serviceGroups] = await Promise.all([
          models.ServiceDevicePolicies.findAll({
            attributes: [
              'device_policy_id',
              'service_id'
            ],
            where: {
              service_id: {
                [Op.in]: serviceIds
              },
              device_policy_id: {
                [Op.in]: _policyIds,
                [Op.ne]: policy.id
              }
            },
            raw: true
          }),
          models.ServiceGroupDevicePolicies.findAll(
            {
              attributes: [
                'device_policy_id',
                'service_group_id'
              ],
              where: {
                service_group_id: {
                  [Op.in]: serviceGroupIds
                },
                device_policy_id: {
                  [Op.in]: _policyIds,
                  [Op.ne]: policy.id
                }
              },
              raw: true
            }
          )
        ]);
        
        if(_services.length > 0 || _serviceGroups.length > 0){
          const policy_ids = [
            ..._services.map(_c => _c.device_policy_id), 
            ..._serviceGroups.map(_cg => _cg.device_policy_id)
          ];
  
          if(policy_ids.length > 0){
            const conflicting_policies = await DevicePolicies.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policy_ids
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
  
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    }


  }

  return DevicePolicies;
};
