const httpErrors = require('http-errors');
const _ = require('lodash');

('use strict');
module.exports = (sequelize, DataTypes) => {
  const RoutingPolicies = sequelize.define(
    'RoutingPolicies',
    {
      id: {
        allowNull: false,
        autoIncrement: true,
        primaryKey: true,
        type: DataTypes.INTEGER
      },
      name: {
        type: DataTypes.STRING
      },
      is_enabled: {
        type: DataTypes.BOOLEAN
      },
      all_gateways: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      all_services: {
        type: DataTypes.BOOLEAN,
        defaultValue: false
      },
      createdAt: {
        allowNull: false,
        type: DataTypes.DATE
      },
      updatedAt: {
        allowNull: false,
        type: DataTypes.DATE
      }
    },
    {}
  );
  RoutingPolicies.associate = function(models) {
    // associations can be defined here
    RoutingPolicies.hasMany(models.GatewayRoutingPolicies, {
      foreignKey: 'routing_policy_id',
      as: 'GRPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });
    RoutingPolicies.hasMany(models.ServiceRoutingPolicies, {
      foreignKey: 'routing_policy_id',
      as: 'SRPolicies',
      onDelete: 'CASCADE'
    });
    RoutingPolicies.hasMany(models.GatewayGroupRoutingPolicies, {
      foreignKey: 'routing_policy_id',
      as: 'GGroupRPolicies',
      onDelete: 'CASCADE'
    });
    RoutingPolicies.hasMany(models.ServiceGroupRoutingPolicies, {
      foreignKey: 'routing_policy_id',
      as: 'SGroupRPolicies',
      onDelete: 'CASCADE'
    });

    RoutingPolicies.belongsToMany(models.Gateway, {
      foreignKey: 'routing_policy_id',
      through: models.GatewayRoutingPolicies
    });
    RoutingPolicies.belongsToMany(models.Service, {
      foreignKey: 'routing_policy_id',
      through: models.ServiceRoutingPolicies
    });
    RoutingPolicies.belongsToMany(models.GatewayGroups, {
      foreignKey: 'routing_policy_id',
      through: models.GatewayGroupRoutingPolicies
    });
    RoutingPolicies.belongsToMany(models.ServiceGroup, {
      foreignKey: 'routing_policy_id',
      through: models.ServiceGroupRoutingPolicies
    });
  };

  RoutingPolicies.addScope('getEnablePolicy', {
    where: {
      is_enabled: 1
    }
  });

  //Hooks

  RoutingPolicies.addHook('beforeCreate', checkIfPolicyExist);
  RoutingPolicies.addHook('beforeUpdate', checkIfPolicyExist);
  /**
   *
   * @param {Object} policy
   */
  /**
   * Check if the name of the policy already exist
   * @param {string} policy
   */
  async function checkIfPolicyExist(policy, options) {

    const models = require('../models');
    const isUpdate = options.updateData ? true : false;


    const all_gateways = isUpdate ? options.updateData.all_gateways : policy.all_gateways; 
    const all_services = isUpdate ? options.updateData.all_services : policy.all_services;

    policy.dataValues.name = _.upperFirst(
      _.toLower(policy.dataValues.name.trim())
    );
    if (!policy.dataValues.name)
      throw new httpErrors.UnprocessableEntity('Name is required');

    const name = policy.dataValues.name;
    if (
      options.fields.includes('name') &&
      policy.name !== policy.previous('name')
    ) {
      const result = await RoutingPolicies.findOne({
        where: {
          name
        }
      });

      if (result)
        throw new httpErrors.Conflict('Policy with this name already exist');
    }



    if(all_services && all_gateways){
      const conflicting_policy = await RoutingPolicies.findOne({
        where: {
          all_gateways: true,
          all_services: true,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
  
      if(conflicting_policy){
        throw new httpErrors.Conflict(`Policy already exists with same configuration. Existing policy name is ${conflicting_policy.name}.`);
      }
    }

    if(all_services && !all_gateways){
      const gatewayIds = isUpdate ? options.updateData.gateway : policy.GRPolicies.map(_gateway => _gateway.gateway_id);
      const gatewayGroupIds = isUpdate ? options.updateData.gatewayGroup : policy.GGroupRPolicies.map(_policy => _policy.gateway_group_id);

      const [conflicting_gateways, conflicting_gateway_groups] = await Promise.all([
        models.GatewayRoutingPolicies.findAll({
          attributes: [
            'routing_policy_id',
            'gateway_id'
          ],
          where: {
            gateway_id:{
              [Op.in]: gatewayIds
            },
            routing_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.GatewayGroupRoutingPolicies.findAll({
          attributes: [
            'routing_policy_id',
            'gateway_group_id'
          ],
          where: {
            gateway_group_id:{
              [Op.in]: gatewayGroupIds
            },
            routing_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        })
      ])

      if(conflicting_gateways.length > 0 || conflicting_gateway_groups.length > 0){
        const policyIds = [
          ...conflicting_gateways.map(_c => _c.routing_policy_id), 
          ...conflicting_gateway_groups.map(_cg => _cg.routing_policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await RoutingPolicies.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }else{
        const RoutingPolicyWithAllGateways = await RoutingPolicies.findAll({
          where: {
            all_gateways: true,
            all_services: false
          }
        });
  
        if(RoutingPolicyWithAllGateways.length){
          const names = RoutingPolicyWithAllGateways.map(_p => _p.name);
          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

    }

    if(!all_services && all_gateways){
      const serviceIds = isUpdate ? options.updateData.service : policy.SRPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupRPolicies.map(_policy => _policy.service_group_id);
      
      const [services, serviceGroups] = await Promise.all([
        models.ServiceRoutingPolicies.findAll({
          attributes: [
            'routing_policy_id',
            'service_id'
          ],
          where: {
            service_id: {
              [Op.in]: serviceIds
            },
            routing_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.ServiceGroupRoutingPolicies.findAll(
          {
            attributes: [
              'routing_policy_id',
              'service_group_id'
            ],
            where: {
              service_group_id: {
                [Op.in]: serviceGroupIds
              },
              routing_policy_id: {
                [Op.ne]: policy.id
              }
            },
            raw: true
          }
        )
      ]);

      if(services.length > 0 || serviceGroups.length > 0){
        const policyIds = [
          ...services.map(_c => _c.routing_policy_id), 
          ...serviceGroups.map(_cg => _cg.routing_policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await RoutingPolicies.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

      const allGatewayPolicy = await models.RoutingPolicies.findAll({
        where: {
          all_services: true,
          all_gateways: false,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
      
      if(allGatewayPolicy.length > 0){
        const names = allGatewayPolicy.map(_p => _p.name);
        throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
      }
    
    }

    if(!all_services && !all_gateways){
      const gatewayIds = isUpdate ? options.updateData.gateway : policy.GRPolicies.map(_gateway => _gateway.gateway_id);
      const gatewayGroupIds = isUpdate ? options.updateData.gatewayGroup : policy.GGroupRPolicies.map(_policy => _policy.gateway_group_id);
      const serviceIds = isUpdate ? options.updateData.service :  policy.SRPolicies.map(_policy => _policy.service_id);
      const serviceGroupIds = isUpdate ? options.updateData.serviceGroup : policy.SGroupRPolicies.map(_policy => _policy.service_group_id);

      const [gateways, gatewayGroups] = await Promise.all([
        models.GatewayRoutingPolicies.findAll({
            attributes: [
              'routing_policy_id',
              'gateway_id'
            ],
            where: {
              gateway_id: {
                [Op.in]: gatewayIds
              },
              routing_policy_id: {
                [Op.ne]: policy.id
              }
            },
            raw: true
        }),
        models.GatewayGroupRoutingPolicies.findAll({
          attributes: [
            'routing_policy_id',
            'gateway_group_id'
          ],
          where: {
            gateway_group_id: {
              [Op.in]: gatewayGroupIds
            },
            routing_policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        })
      ]); 
      
      const policyIds = gateways.map(gateway => {
        return gateway.routing_policy_id;
      });

      gatewayGroups.forEach(group => {
        policyIds.push(group.routing_policy_id);
      })
      
      const [services, serviceGroups] = await Promise.all([
            models.ServiceRoutingPolicies.findAll({
              attributes: [
                'routing_policy_id',
                'service_id'
              ],
              where: {
                service_id: {
                  [Op.in]: serviceIds
                },
                routing_policy_id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                }
              },
              raw: true
            }),
            models.ServiceGroupRoutingPolicies.findAll(
              {
                attributes: [
                  'routing_policy_id',
                  'service_group_id'
                ],
                where: {
                  service_group_id: {
                    [Op.in]: serviceGroupIds
                  },
                  routing_policy_id: {
                    [Op.in]: policyIds,
                    [Op.ne]: policy.id
                  }
                },
                raw: true
              }
            )
          ]);
    
        if(services.length > 0 || serviceGroups.length > 0){
          const policyIds = [
            ...services.map(_c => _c.routing_policy_id), 
            ...serviceGroups.map(_cg => _cg.routing_policy_id)
          ];
  
          if(policyIds.length > 0){
            const conflicting_policies = await RoutingPolicies.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policyIds
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
  
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    
        if(policyIds.length){
            const allServicePolicy = await models.RoutingPolicies.findAll({
              where: {
                id: {
                  [Op.in]: policyIds,
                  [Op.ne]: policy.id
                },
                all_services: true,
                all_gateways: false
              }
            });
            
            if(allServicePolicy.length > 0){
              const names = allServicePolicy.map(_p => _p.name);
              throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
            }
        }

        const allGatewayPolicy = await models.RoutingPolicies.findAll({
          where: {
            all_services: false,
            all_gateways: true,
            id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        });

        _policyIds = allGatewayPolicy.map(_policy => _policy.id);

        const [_services, _serviceGroups] = await Promise.all([
          models.ServiceRoutingPolicies.findAll({
            attributes: [
              'routing_policy_id',
              'service_id'
            ],
            where: {
              service_id: {
                [Op.in]: serviceIds
              },
              routing_policy_id: {
                [Op.in]: _policyIds,
                [Op.ne]: policy.id
              }
            },
            raw: true
          }),
          models.ServiceGroupRoutingPolicies.findAll(
            {
              attributes: [
                'routing_policy_id',
                'service_group_id'
              ],
              where: {
                service_group_id: {
                  [Op.in]: serviceGroupIds
                },
                routing_policy_id: {
                  [Op.in]: _policyIds,
                  [Op.ne]: policy.id
                }
              },
              raw: true
            }
          )
        ]);
        
        if(_services.length > 0 || _serviceGroups.length > 0){
          const policy_ids = [
            ..._services.map(_c => _c.routing_policy_id), 
            ..._serviceGroups.map(_cg => _cg.routing_policy_id)
          ];
  
          if(policy_ids.length > 0){
            const conflicting_policies = await RoutingPolicies.findAll({
              attributes: ['name'],
              where: {
                id: {
                  [Op.in]: policy_ids
                }
              }
            });
  
            const names = conflicting_policies.map(_p => _p.name);
  
            throw new httpErrors.Conflict(`There is conflict in service(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
          }
        }
    }


  }

  return RoutingPolicies;
};
