const _ = require('lodash');
const moment = require('moment');
const httpErrors = require('http-errors');

'use strict';
module.exports = (sequelize, DataTypes) => {
  const ServerBridge = sequelize.define('ServerBridge', {
    name: DataTypes.STRING,
    clientId:DataTypes.INTEGER,
    isEnable:DataTypes.BOOLEAN,
    email_token:DataTypes.STRING,
    email_token_used:DataTypes.BOOLEAN,
    email_token_expiry:DataTypes.DATE,
    type: DataTypes.STRING,
    device_id: DataTypes.INTEGER
  }, {});


  //Instance level method defined

  ServerBridge.prototype.getClientOnlineStatus = async function() {
    const models = require('../models');
    
    const clientId = this.get('clientId');

    if(!clientId) return false;

    try{

      const client = await models.Client.findOne({
        where:{
          id:clientId
        },
        attributes:['heart_beat','heart_beat_sent_at']
      });

      if(!client) return false;

      const { heart_beat, heart_beat_sent_at } = client;

      if( !heart_beat ) return false;
      
      const allowedTimeInUtc = moment().subtract(120, 'seconds');

      const diffInSeconds = moment(heart_beat_sent_at)
        .utc()
        .diff(allowedTimeInUtc, 'second');

      return diffInSeconds >= 0 ? true : false;

    }catch(error){

      return false;

    }
  }


  ServerBridge.associate = function(models) {
    // associations can be defined here

    ServerBridge.belongsTo(models.Client,{
      foreignKey:'clientId'
    })
    ServerBridge.hasMany(models.ServerBridgeDeviceId, { foreignKey:"serverbridgeId", as:"serverBridgeIds" })
    ServerBridge.hasOne(models.DeviceInventories, { foreignKey: "device_id"})
  };


  ServerBridge.addHook('beforeCreate',checkName);
  ServerBridge.addHook('beforeUpdate',checkName);


  async function checkName( bridge,options ){
    
    bridge.dataValues.name = _.upperFirst(
      _.toLower(bridge.dataValues.name.trim())
    );
    
    const name = bridge.dataValues.name;
    const type = bridge.dataValues.type;

    if(bridge.previous('name') !== name){
      //find if exists
      const bridge = await ServerBridge.findOne({
        where:{
          name,
          type
        },
        attributes:['id'],
        limit:1
      });

      if(bridge){
        throw new httpErrors.Conflict('Server Bridge with this name already exist');
      }
    }

  }


  return ServerBridge;
};