const httpErrors = require('http-errors');
const _ = require('lodash');

'use strict';

const { ConstraintViolationError } = require('ldapjs');
const { Console } = require('winston/lib/winston/transports');

module.exports = function(sequelize, DataTypes) {
  var SoftwarePolicies = sequelize.define('SoftwarePolicies', {
    name: DataTypes.STRING,
    grace_period: DataTypes.INTEGER,
    block_services:DataTypes.BOOLEAN,
    all_clients: DataTypes.BOOLEAN,
    all_services: DataTypes.BOOLEAN
  });

  SoftwarePolicies.associate = function(models) {

    SoftwarePolicies.hasMany(models.ClientSoftwarePolicies, {
      foreignKey: 'policy_id',
      as: 'CSPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });

    SoftwarePolicies.hasMany(models.ClientGroupSoftwarePolicies, {
      foreignKey: 'policy_id',
      as: 'CGroupSPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });

    SoftwarePolicies.hasMany(models.ServiceGroupSoftwarePolicies, {
      foreignKey: 'policy_id',
      as: 'SGroupSPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });
    
    SoftwarePolicies.hasMany(models.ServiceSoftwarePolicies, {
      foreignKey: 'policy_id',
      as: 'SSPolicies',
      onDelete: 'CASCADE',
      hooks: true
    });

    SoftwarePolicies.belongsToMany(models.Client, {
      through: models.ClientSoftwarePolicies,
      foreignKey: 'policy_id'
    });
 
    SoftwarePolicies.belongsToMany(models.Service, {
      through: models.ServiceSoftwarePolicies,
      foreignKey: 'policy_id'
    });
 
    SoftwarePolicies.belongsToMany(models.Group, {
      foreignKey: 'policy_id',
      through: models.ClientGroupSoftwarePolicies
    });

    SoftwarePolicies.belongsToMany(models.ServiceGroup, {
      foreignKey: 'policy_id',
      through: models.ServiceGroupSoftwarePolicies
    });


  };

    //Hooks

    SoftwarePolicies.addHook('beforeCreate', checkIfPolicyExist);
    SoftwarePolicies.addHook('beforeUpdate', checkIfPolicyExist);

      /**
   * Check if the name of the policy already exist
   * @param {string} policy
   */
  async function checkIfPolicyExist(policy, options) {

    policy;

    const models = require('../models');
    const isUpdate = options.updateData ? true : false;


    const all_clients = isUpdate ? options.updateData.all_clients : policy.all_clients; 

    policy.dataValues.name = _.upperFirst(
      _.toLower(policy.dataValues.name.trim())
    );
    if (!policy.dataValues.name)
      throw new httpErrors.UnprocessableEntity('Name is required');

    const name = policy.dataValues.name;
    if (
      options.fields.includes('name') &&
      policy.name !== policy.previous('name')
    ) {
      const result = await SoftwarePolicies.findOne({
        where: {
          name
        }
      });

      if (result)
        throw new httpErrors.Conflict('Policy with this name already exist');
    }



    if(all_clients){
      const conflicting_policy = await SoftwarePolicies.findOne({
        where: {
          all_clients: true,
          id: {
            [Op.ne]: policy.id
          }
        }
      });
  
      if(conflicting_policy){
        throw new httpErrors.Conflict(`Policy already exists with same configuration. Existing policy name is ${conflicting_policy.name}.`);
      }
    }

    if(!all_clients){
      const clientIds = isUpdate ? options.updateData.client : policy.CSPolicies.map(_client => _client.client_id);
      const clientGroupIds = isUpdate ? options.updateData.clientGroup : policy.CGroupSPolicies.map(_policy => _policy.client_group_id);

      const [conflicting_clients, conflicting_client_groups] = await Promise.all([
        models.ClientSoftwarePolicies.findAll({
          attributes: [
            'policy_id',
            'client_id'
          ],
          where: {
            client_id:{
              [Op.in]: clientIds
            },
            policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        }),
        models.ClientGroupSoftwarePolicies.findAll({
          attributes: [
            'policy_id',
            'client_group_id'
          ],
          where: {
            client_group_id:{
              [Op.in]: clientGroupIds
            },
            policy_id: {
              [Op.ne]: policy.id
            }
          },
          raw: true
        })
      ])

      if(conflicting_clients.length > 0 || conflicting_client_groups.length > 0){
        const policyIds = [
          ...conflicting_clients.map(_c => _c.policy_id), 
          ...conflicting_client_groups.map(_cg => _cg.policy_id)
        ];

        if(policyIds.length > 0){
          const conflicting_policies = await SoftwarePolicies.findAll({
            attributes: ['name'],
            where: {
              id: {
                [Op.in]: policyIds
              }
            }
          });

          const names = conflicting_policies.map(_p => _p.name);

          throw new httpErrors.Conflict(`There is conflict in client(s). Conflicting ${names.length > 1 ? 'policies are' : 'policy is'} ${names.join(', ')}.`);
        }
      }

    }

  }

  return SoftwarePolicies;
};
