
const { authenticator } = require('otplib');
const qrcode = require('qrcode');
const { promisify } = require('util');
const toDataURL = promisify(qrcode.toDataURL);

authenticator.options = {
  window: 1
}

'use strict';

module.exports = function(sequelize, DataTypes) {
  var User = sequelize.define('User', {
    firstName: DataTypes.STRING,
    lastName: DataTypes.STRING,
    email: DataTypes.STRING,
    password: DataTypes.STRING,
    is_active: DataTypes.BOOLEAN,
    createdById: DataTypes.INTEGER,
    updatedById: DataTypes.INTEGER,
    reset_password_token: DataTypes.STRING,
    reset_password_expires: DataTypes.DATE,
    reset_password_time: DataTypes.DATE,
    wrong_login_attempts: DataTypes.INTEGER,
    is_locked: DataTypes.BOOLEAN,
    email_verified: DataTypes.BOOLEAN,
    email_verification_link: DataTypes.STRING,
    email_verification_link_expiry: DataTypes.DATE,
    is_mfa_assigned: DataTypes.BOOLEAN,
    otp_image_data: DataTypes.STRING,
    public_key: DataTypes.STRING,
    secret_key: DataTypes.STRING,
    otp_secret: DataTypes.STRING
  });

  User.prototype.toJSON = function() {
    var values = Object.assign({}, this.get());

    delete values.password;
    return values;
  };

  User.prototype.generateOtp = async function() {
    await generateOtp(this);
    return this.save();
  }

  User.prototype.verifyOtp = function(token) {
    if (!this.otp_secret) return false;
    const check = authenticator.checkDelta(token, this.otp_secret);
    return Number.isInteger(check);
  }

  User.addHook('beforeCreate', user => {
    user.dataValues.email_verified = 0;
  });

  User.associate = function(models) {
    User.belongsToMany(models.Role, {
      through: 'UserRole',
      foreignKey: 'userId'
    });
    User.hasMany(models.AdminAccessLogs, { foreignKey: 'userId' });
  };

  async function generateOtp(user, options) {

    if(user.otp_image_data) {
      return;
    };

    user.otp_secret = authenticator.generateSecret();
    const otpauth = authenticator.keyuri(user.email, 'INVISILY ADMIN', user.otp_secret);
    const img = await toDataURL(otpauth);
    user.otp_image_data = img.split(';base64,').pop().slice(0, -5);
    
  }


  return User;
};
