const ldapjs = require('ldapjs');

const PagedControl = require('ldapjs/lib/controls/paged_results_control');

class LdapManager {
  constructor() {
    this.connection = null;
    this.connectionId = 0;
    this.validConnection = false;
    this.settings = null;
    this.SIZE_LIMIT = 1000;
    this.PREFFERED_PAGE_SIZE = 100;
    this.keepAliveInterval = null;
  }

  createLdapConnection(ldap) {
    const ldapConnect = new Promise((resolve, reject) => {
      this.validConnection = false;

      if (ldap) {
        console.log('LDAPManager: Found LDAP settings');
        console.log(ldap.LDAP_connectTimeout);
        const ldapOptions = {
          url: ldap.LDAP_url,
          connectTimeout: ldap.LDAP_connectTimeout,
          idleTimeout: ldap.LDAP_timeout,
          timeout: 30000, // Single operation should take less than 30 seconds
          tlsOptions: {
            cert: ldap.LDAP_cert,
            ca: ldap.LDAP_ca
          }
        };

        this.connection = ldapjs.createClient(ldapOptions);

        console.log('LDAPManager: Connection Initiated');

        this.connection.on('error', err => {
          this.invalidateLdapConnection();

          console.error('LDAPManager: Connection Error: ', err.message);
          reject(err);
        });

        this.connection.on('socketTimeout', () => {
          this.invalidateLdapConnection();

          console.log('LDAPManager: Connection Timeout');
        });

        this.connection.on('connectError', err => {
          this.invalidateLdapConnection();

          console.error('LDAPManager: Connection Failed: ', err.message);
          reject(err);
        });

        this.connection.on('idle', () => {
          this.invalidateLdapConnection();
          console.log('LDAPManager: Connection Idle. Unbinding.');

          this.connection.unbind(() => {
            console.log('LDAPManager: Unbound.');
          });
        });

        this.connection.on('close', () => {
          this.invalidateLdapConnection();
          console.log('LDAPManager: Connection Closed.');
        });

        this.connection.on('end', () => {
          this.invalidateLdapConnection();
          console.log('LDAPManager: Connection Ended.');
        });

        this.connection.on('connect', () => {
          console.log('LDAPManager: Connected');
          this.connectionId = parseInt(Math.random() * 10000, 10);
          this.settings = ldap;

          resolve();
        });
      } else {
        reject({ status: 'SettingsNotFound' });
      }
    });

    return ldapConnect
      .then(() => {
        return this.bindDefaultUser();
      })
      .then(() => {
        this.validConnection = true;
        return { status: true };
      });
  }

  reuseOrCreateLdapConnection(ldap) {
    if (!this.validConnection) {
      return this.createLdapConnection(ldap);
    } else {
      return Promise.resolve();
    }
  }

  getBindError(err) {

    console.log('binding error ', err.code)
    const ADErrors = {
      // IF AD returns key as code then send value to client as response.
      '525': 1317,
      '52e': 1326,
      '530': 1328,
      '531': 1329,
      '532': 1330,
      '533': 1331,
      '534': 1332,
      '701': 1793,
      '773': 1907,
      '775': 1909,
    }
    let adErrorFound = null;

    if(err.toString) {
      console.log('inside to string')
      Object.entries(ADErrors).forEach(([ADError, errorCode]) => {
        console.log('chekcing fo r', ADError, errorCode, err.toString(), err.toString().includes(ADError))
          if(err.toString() && err.toString().includes(ADError)) {
            adErrorFound = {
              "name": "ADCustomError",
              "errorCode": errorCode
            }
          }
      })
    }

    if(adErrorFound) {
      return adErrorFound;
    }

    if (err instanceof ldapjs.InvalidCredentialsError) {
      return { status: 'InvalidCredentials' };
    } else if (err instanceof ldapjs.InappropriateAuthenticationError) {
      console.error('LDAPManager: Inappropriate Authentication');
      return { status: 'InappropriateAuthenticationError' };
    } else {
      console.error('LDAPManager: ', err.message);
      return err;
    }
  }

  bindDefaultUser() {
    return new Promise((resolve, reject) => {
      this.connection.bind(
        this.settings.LDAP_user,
        this.settings.LDAP_password,
        err => {
          if (err) {
            return reject(this.getBindError(err));
          }
          console.log(
            `LDAPManager: Bind successful with user ${this.settings.LDAP_user}`
          );
          resolve(true);
        }
      );
    });
  }

  bindUser(username, password) {
    return new Promise((resolve, reject) => {
      this.connection.bind(username, password, err => {
        if (err) {
          // if (err) {
          //   console.log('Error in ldap auth ', err);
          //   return resolve(false);
          // }
          // if (err instanceof ldapjs.InvalidCredentialsError) {
          //   return resolve(false);
          // } else {
            return reject(this.getBindError(err));
          // }
        } else {
          console.log(
            `LDAPManager: Bind successful with user ${this.settings.LDAP_user}`
          );
          resolve(true);
        }
      });
    });
  }

  do(opType, ...args) {
    const clientOp = this.connection[opType];
    const managerOp = this[opType];

    if (this.validConnection && clientOp) {
      return clientOp.apply(this.connection, args);
    } else if (this.validConnection && managerOp) {
      return managerOp.apply(this, args);
    }
  }

  skipRequest(options, pageSize) {
    return cookie =>
      new Promise((resolve, reject) => {
        const skipControls = [
          new PagedControl({
            value: {
              size: pageSize,
              cookie: cookie || null
            }
          })
        ];

        this.do(
          'search',
          this.settings.LDAP_dc,
          options,
          skipControls,
          (err, res) => {
            if (err) {
              console.log(err && (err.message || err));
              reject({ status: 'OperationFailed' });
              return;
            }

            res.on('error', err => {
              console.log(err && (err.message || err));
              reject({ status: 'OperationFailed' });
              return;
            });

            res.on('end', result => {
              const pagedResponse = result.controls.filter(
                control => control instanceof PagedControl
              )[0];

              if (pagedResponse) {
                resolve(pagedResponse.value.cookie);
              } else {
                reject({ status: 'OperationFailed' });
                return;
              }
            });
          }
        );
      });
  }

  skipTo(options, pageNumber, pageSize) {
    return new Promise((resolve, reject) => {
      const totalSkipCount = pageNumber * pageSize;
      const skipPages = parseInt(totalSkipCount / this.SIZE_LIMIT, 10) + 1;
      const lastPageSize = totalSkipCount % this.SIZE_LIMIT;

      let chunks = Array(skipPages).fill(this.SIZE_LIMIT);

      if (lastPageSize !== 0) {
        chunks[skipPages - 1] = lastPageSize;
      } else {
        chunks.pop();
      }

      chunks = chunks.map(size => this.skipRequest(options, size));

      let chunk$ = Promise.resolve();

      chunks.forEach(skip => {
        chunk$ = chunk$.then(cookie => skip(cookie));
      });

      chunk$.then(resolve).catch(reject);
    });
  }

  keepLdapConnectionAlive() {
    this.keepAliveInterval = setInterval(() => {
      console.log('LDAPManager: Keeping Alive');
      this.do(
        'search',
        this.settings.LDAP_dc,
        [],
        [
          new PagedControl({
            value: {
              cookie: null,
              size: 1
            }
          })
        ],
        (err, res) => {
          if (err) {
            console.log('LDAPManager: KeepAlive Error: ', err);
          }

          res.on('error', err => {
            console.log('LDAPManager: KeepAlive Error: ', err);
          });
        }
      );
    }, 60000);
  }

  closeLdapConnection() {
    return new Promise((resolve, reject) => {
      console.log('LDAPManager: Disconnecting');

      if (this.validConnection && this.connection) {
        this.validConnection = false;
        this.connection = null;
        this.settings = null;
        console.log('LDAPManager: Disconnected');
      }

      resolve();
    });
  }

  invalidateLdapConnection() {
    this.validConnection = false;

    if (this.keepAliveInterval) {
      clearInterval(this.keepAliveInterval);
    }
  }
}

module.exports = LdapManager;
